library(DESeq2)
library(TxDb.Hsapiens.UCSC.hg19.knownGene)
library(Rsubread)
library(org.Hs.eg.db)

library(ggplot2)
library(ggrepel)
library("RColorBrewer")
library(pheatmap)

########################################################################################################
### RNAseq DE

workplace = "office"

if(workplace == "home"){
  
path.fig = "D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/counts"
path.dir = "D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/counts"

}

if(workplace == "office"){
  
  path.fig = "C:/Users/ab998/OneDrive - University of Exeter/TDP43 project JG/RNA seq Bulk 2023/Illumina/counts"
  path.dir = "C:/Users/ab998/OneDrive - University of Exeter/TDP43 project JG/RNA seq Bulk 2023/Illumina/counts"
  
}

dds = readRDS(paste(path.dir, "/TDP43KI-model-DESeq2.rds", sep=""))

vsd = vst(dds, blind=FALSE)
sampleDists <- dist(t(assay(vsd)))

sampleDistMatrix <- as.matrix(sampleDists)
#rownames(sampleDistMatrix) <- paste(vsd$condition, vsd$type, sep="-")
#colnames(sampleDistMatrix) <- NULL
colors <- colorRampPalette( rev(brewer.pal(9, "Blues")) )(255)

tiff(paste(path.fig, "/TDP43KI-model-cluster.tiff", sep=""), res=300, compression = "jpeg", width = 8, height = 8, unit = "in")
pheatmap(sampleDistMatrix,
         clustering_distance_rows=sampleDists,
         clustering_distance_cols=sampleDists,
         col=colors)

dev.off()

pdf(paste(path.fig, "/TDP43KI-model-cluster.pdf", sep=""))
pheatmap(sampleDistMatrix,
         clustering_distance_rows=sampleDists,
         clustering_distance_cols=sampleDists,
         col=colors)

dev.off()

### PCA

plotPCA(vsd, intgroup=c("genotype", "condition"), returnData = F)

pca.df = plotPCA(vsd, intgroup=c("genotype", "condition"), returnData = TRUE)
pca.df$treatment = "Control"
pca.df$treatment[which(pca.df$condition == "nes")] = "NES"
pca.df$group = paste(pca.df$genotype, pca.df$treatment, sep=" ")

p = ggplot(pca.df) +
  aes(PC1, PC2, color = group) + 
  geom_point(size = 3) + 
  coord_fixed() +
  xlab("PC1: 90%")+ 
  ylab("PC2: 6%") +
  theme_bw() +
  theme(axis.text=element_text(size=16),
        axis.title = element_text(size=16),
        legend.text = element_text(size=16),
        legend.position = "bottom",
        legend.title = element_blank(),
        panel.grid.minor = element_blank()) +
  guides(colour = guide_legend(override.aes = list(size=4))) +
  theme(aspect.ratio = 0.5)

tiff(paste(path.fig, "/TDP43KI-model-PCA.tiff", sep=""), res=300, compression = "jpeg", width = 6, height = 6, unit = "in")
p
dev.off()

pdf(paste(path.fig, "/TDP43KI-model-PCA.pdf", sep=""))
p
dev.off()

#### volcano plot

df = read.table(paste(path.dir, "TDP43KI-model-DESeq2.txt", sep="/"), sep="\t", header=T)

p.thresh = 0.01
fc.thresh = log2(2)

up <-  df$symbol[which(df$padj < p.thresh & df$log2FoldChange > fc.thresh)] 
down <-  df$symbol[which(df$padj < p.thresh & df$log2FoldChange < -1*fc.thresh)] 

mycolors <- c("blue", "red", "gray")
names(mycolors) <- c("Down", "Up", "NC")

df$de <-  "NC"
df$de[match(up, df$symbol)] <-  "Up"
df$de[match(down, df$symbol)] <-  "Down"

df <- na.omit(df)

goi = c("STMN2", "UNC13A", "ELAVL3", "PFKP", "CYFIP2", "ACTL6B",  "CELF5")
df.subset = df[match(goi, df$symbol),]

p = ggplot(data=df) +
  geom_point(aes(x=log2FoldChange, y=-1*log10(padj), color=de, size=de)) + 
  geom_hline(yintercept= -log10(p.thresh), col="magenta") +
  scale_colour_manual(values = mycolors) +
  scale_size_manual(values = c(2, 1, 2)) +
  scale_x_continuous(limits = c(-4, 4), expand = c(0, 0))



p = p +  theme_bw() + 
  theme(legend.title=element_blank(), legend.text = element_text(colour="black", size = 20), legend.position = "right") +
  guides(colour = guide_legend(override.aes = list(size=4))) + 
  theme(axis.text.x = element_text(face="bold", color="black", size=16, angle=0)) +
  theme(axis.text.y = element_text(face="bold", color="black", size=16, angle=0)) +
  labs(x="log2(Fold change)", y="-1*log10(p-value)") +
  theme(axis.title = element_text(size = 20)) +
  theme(aspect.ratio = 1)


q = p +  geom_text_repel(data = df.subset, 
                         aes(x = log2FoldChange, y =  -1*log10(padj), label=symbol), size = 6,
                         nudge_y=0.5, hjust=1.5, direction="x",
                         segment.color="green", segment.size=1.0,
                         min.segment.length = 0.2)

q

tiff(paste(path.fig, "/TDP43KI-model-volcano.tiff", sep=""), res=300, compression = "jpeg", width = 8, height = 8, unit = "in")
q
dev.off()


### GO bar plot

df = read.table(paste(path.dir, "TDP43KI.deseq.gseaGO.BP.txt", sep="/"), sep="\t", header=T, quote="")

df$Description[which(df$p.adjust < 0.01)]

poi.dw = c("synaptic vesicle cycle", "neurotransmitter secretion", "neurotransmitter transport", "axon development", "regulation of trans-synaptic signaling" )
poi.up = c("protein targeting to ER", "nuclear-transcribed mRNA catabolic process, nonsense-mediated decay",
           "translational initiation", "ribosome assembly", "ribonucleoprotein complex assembly")


poi.up.pval = -1*log10(df$p.adjust[match(poi.up, df$Description)])
poi.dw.pval =  -1*log10(df$p.adjust[match(poi.dw, df$Description)])

poi.dw = c("synaptic vesicle cycle", "neurotransmitter secretion", "neurotransmitter transport", "axon development          ", "regulation of trans-synaptic signaling                                     ")
poi.up = c("protein targeting to ER", "  nonsense-mediated decay", "translational initiation", 
           "                                   mRNA catabolic process", "                              ribosome assembly")

go.up.df = data.frame("pathway" = poi.up, "pval" = poi.up.pval)
go.dw.df = data.frame("pathway" = poi.dw, "pval" = poi.dw.pval)

go.up.df$pathway = factor(go.up.df$pathway, levels = go.up.df$pathway)
go.dw.df$pathway = factor(go.dw.df$pathway, levels = go.dw.df$pathway)

### plot up
tiff(paste(path.fig, "/TDP43KI-model-up-genes-GO.tiff", sep=""), res=300, compression = "jpeg", width = 9, height = 6, unit = "in")

ggplot(go.up.df) +
  
  geom_col(aes(x = pathway, y = pval), fill = "khaki1", color = "black", width = 0.4, position = position_dodge(width=0.9)) +
  coord_flip() +
  scale_x_discrete(name = "") +
  scale_y_continuous(name = "-Log(P adjusted)", 
                     limits = c(0, 50),
                     expand = c(0, 0)) +
  ggtitle("Up-regulated genes") +
  
  theme(panel.border = element_blank(),
        axis.line = element_line(color = 'black', 
                                 linewidth = 0.5),
        axis.text.y = element_blank(),
        axis.ticks.y = element_blank(),
        axis.title.x = element_text(size = 24, face = "bold"),
        axis.text.x = element_text(size = 24, color="black"),
        plot.title = element_text(size = 36),
        aspect.ratio = 0.5) +
  
  geom_text(aes(x = pathway, y = pval/2, label = pathway), size= 8) +
  theme(plot.margin = unit(c(0.1, 0.5, 0.1, 0.1), "cm"))

dev.off()

### plot down

tiff(paste(path.fig, "/TDP43KI-model-down-genes-GO.tiff", sep=""), res=300, compression = "jpeg", width = 9, height = 6, unit = "in")

ggplot(go.dw.df) +
  
  geom_col(aes(x = pathway, y = pval), fill = "deepskyblue", color = "black", width = 0.4, position = position_dodge(width=1.0)) + 
  
  coord_flip() +
  
  scale_x_discrete(name = "", position = "top") + # Add axis to right side of plot (b/c of flipped coordinates, this is the x-axis)
  scale_y_reverse(name = "-Log(P adjusted)",   # Change the bottom axis to be in reverse ((b/c of flipped coordinates, this is the y-axis). Adjust the breaks as needed.
                  breaks = c(0, 2, 4, 6, 8, 10),
                  limits = c(10,0),
                  expand = c(0, 0)) +
  ggtitle("Down-regulated genes") +
  
  theme(panel.border = element_blank(),
        axis.line = element_line(color = 'black', 
                                 size = 0.5),
        axis.text.y = element_blank(),
        axis.ticks.y = element_blank(),
        axis.title.x = element_text(size = 24, face = "bold"),
        axis.text.x = element_text(size = 24, color="black"),
        plot.title = element_text(size = 36),
        aspect.ratio = 0.6) +
  
  geom_text(aes(x = pathway, y = pval/2, label = pathway), size=8) +
  theme(plot.margin = unit(c(0.1, 0.1, 0.1, 0.5), "cm"))

dev.off()


##########################################################################################
### AS gene GO analysis

if(workplace == "home"){
  
  path.fig = "D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/AS"
  path.dir = "D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/AS"
  
}

if(workplace == "office"){
  
  path.fig = "C:/Users/ab998/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/AS"
  path.dir = "C:/Users/ab998/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/AS"
  
}

library(DOSE)
library(clusterProfiler)
library(pathview)

res.df = read.table(paste("D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/AS/tdp43_leafcutter_merged.txt"), sep="\t",
                    header=T)

res.de.df = read.table("D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/Illumina/Counts/TDP43KI-model-DESeq2.txt", sep="\t",
                    header=T)

allOE_genes <- unique(res.de.df$symbol[which(res.de.df$baseMean > 1)])
length(allOE_genes)
sigOE_genes <- unique(as.character(res.df$genes[res.df$p.adjust< 0.01]))
length(sigOE_genes)

### keyType indicates the id used per gene
### OrgDb indicates human or mouse etc
### ont can be BP or MF or CC or ALL
### pAdjustMethod: use BH as this is less conservative
### qvalueCutoff: threshold for qvalue which can be though of as the FDR

ego <- enrichGO(gene = sigOE_genes, 
                universe = allOE_genes,
                keyType = "SYMBOL",
                OrgDb = org.Hs.eg.db, 
                ont = "BP", 
                pAdjustMethod = "BH", 
                qvalueCutoff = 0.5, 
                pvalueCutoff  = 0.5,
                readable = F)

ego.df <- data.frame(ego)

ego.df[1:20, 1:7]

##########################################################################################
### AIO mirna

if(workplace == "home"){
  
  path.fig = "D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/miRNA"
  path.dir = "D:/OneDrive - University of Exeter/TDP43 project JG/RNA seq/miRNA"
  
}

if(workplace == "office"){
 
  path.fig = "C:/Users/ab998/OneDrive - University of Exeter/TDP43 project JG/RNA seq Bulk 2023/miRNA"
  path.dir = "C:/Users/ab998/OneDrive - University of Exeter/TDP43 project JG/RNA seq Bulk 2023/miRNA"
  
}

sample.all.df = read.table(paste(path.dir, "sample-info-corrected.txt", sep="/"), sep="\t", header = T)

dds = readRDS(paste(path.dir, "/mirna-aio-d40-DESeq2.rds", sep=""))

count.mat = data.frame(counts(dds, normalized = F))
mirID = rownames(count.mat)
count.mat = cbind(mirID, count.mat)
write.table(count.mat, paste(path.dir, "/mirna-aio-d40-counts.txt", sep=""), sep="\t", row.names = F, col.names = T, quote = F)

sample.df = sample.all.df[1:6, ]
sample.df$name = "Control"
sample.df$name[which(sample.df$treatment == "nesm")] = "NES"




vsd = varianceStabilizingTransformation(dds, blind=FALSE)
sampleDists <- dist(t(assay(vsd)))

### clustering

sampleDistMatrix <- as.matrix(sampleDists)
rownames(sampleDistMatrix) <- sample.df$name
colnames(sampleDistMatrix) <- NULL
colors <- colorRampPalette( rev(brewer.pal(9, "Blues")) )(255)

tiff(paste(path.fig, "/AIO-mirna-cluster.tiff", sep=""), res=300, compression = "jpeg", width = 10, height = 10, unit = "in")
pheatmap(sampleDistMatrix,
         clustering_distance_rows=sampleDists,
         clustering_distance_cols=sampleDists,
         col=colors)

dev.off()

pdf(paste(path.fig, "/AIO-mirna-cluster.pdf", sep=""))
pheatmap(sampleDistMatrix,
         clustering_distance_rows=sampleDists,
         clustering_distance_cols=sampleDists,
         col=colors)

dev.off()

### PCA

plotPCA(vsd, intgroup=c("treatment", "replicate"), returnData = F)

pca.df = plotPCA(vsd, intgroup=c("treatment", "replicate"), returnData = TRUE)
pca.df$name = "Control"
pca.df$name[which(pca.df$treatment == "nesm")] = "NES"


tiff(paste(path.fig, "/AIO-mirna-PCA.tiff", sep=""), res=300, compression = "jpeg", width = 6, height = 6, unit = "in")

ggplot(pca.df) +
  aes(PC1, PC2, color = name) + 
  geom_point(size = 3) + 
  coord_fixed() +
  xlab("PC1: 77%")+ 
  ylab("PC2: 9%") +
  theme_bw() +
  theme(axis.text=element_text(size=16),
        axis.title = element_text(size=16),
        legend.text = element_text(size=20),
        legend.position = "bottom",
        legend.title = element_blank(),
        panel.grid.minor = element_blank(),
        aspect.ratio = 1)

dev.off()

pdf(paste(path.fig, "/AIO-mirna-PCA.pdf", sep=""))

ggplot(pca.df) +
  aes(PC1, PC2, color = group) + 
  geom_point(size = 3) + 
  coord_fixed() +
  xlab("PC1: 78%")+ 
  ylab("PC2: 9%") +
  theme_bw() +
  theme(axis.text=element_text(size=16),
        axis.title = element_text(size=16),
        legend.position = "bottom",
        legend.title = element_blank(),
        panel.grid.minor = element_blank(),
        aspect.ratio = 1)

dev.off()

#### volcano plot

df = read.table(paste(path.dir, "mirna-aio-d40-DESeq2.txt", sep="/"), sep="\t", header=T)

p.thresh = 0.01
fc.thresh = 0.58

up <-  df$GeneID[which(df$padj < p.thresh & df$log2FoldChange > fc.thresh)] 
down <-  df$GeneID[which(df$padj < p.thresh & df$log2FoldChange < -1*fc.thresh)] 

mycolors <- c("blue", "red", "gray")
names(mycolors) <- c("Down", "Up", "NC")

df$de <-  "NC"
df$de[match(up, df$GeneID)] <-  "Up"
df$de[match(down, df$GeneID)] <-  "Down"

df <- na.omit(df)

goi = c("hsa-miR-9-5p", "hsa-miR-9-3p", "hsa-miR-218-5p", "hsa-miR-124-5p", "hsa-miR-124-3p")
df.subset = df[match(goi, df$GeneID),]

p = ggplot(data=df) +
  geom_point(aes(x=log2FoldChange, y=-1*log10(padj), color=de, size=de)) + 
  geom_hline(yintercept= -log10(p.thresh), col="magenta") +
  scale_colour_manual(values = mycolors) +
  scale_size_manual(values = c(2, 1, 2)) +
  scale_x_continuous(limits = c(-4, 4), expand = c(0, 0))
 


p = p +  theme_bw() + 
  theme(legend.title=element_blank(), legend.text = element_text(colour="black", size = 20), legend.position = "bottom") +
  guides(colour = guide_legend(override.aes = list(size=4))) + 
  theme(axis.text.x = element_text(face="bold", color="black", size=16, angle=0)) +
  theme(axis.text.y = element_text(face="bold", color="black", size=16, angle=0)) +
  labs(x="log2(Fold change)", y="-1*log10(p-value)") +
  theme(axis.title = element_text(size = 20)) +
  theme(aspect.ratio = 1)


q = p +  geom_text_repel(data = df.subset, 
                         aes(x = log2FoldChange, y =  -1*log10(padj), label=GeneID), size = 6,
                         nudge_y=0.5, hjust=1.5, direction="x",
                         segment.color="green", segment.size=1.0,
                         min.segment.length = 0.2)

q

tiff(paste(path.fig, "/AIO-mirna-volcano.tiff", sep=""), res=300, compression = "jpeg", width = 8, height = 8, unit = "in")
q
dev.off()

############################################################################################################################
### TDP43 KD mirna

sample.df = read.table(paste(path.dir, "sample-info-corrected.txt", sep="/"), sep="\t", header = T)
dds = readRDS(paste(path.dir, "/mirna-TDP43-KD-DESeq2.rds", sep=""))

sample.df = sample.all.df[7:11, ]
sample.df$name = "Control KD"
sample.df$name[which(sample.df$treatment == "cds41")] = "TDP43 KD"

count.mat = data.frame(counts(dds, normalized = F))
mirID = rownames(count.mat)
count.mat = cbind(mirID, count.mat)

write.table(count.mat, paste(path.dir, "/mirna-TDP43-KD-counts.txt", sep=""), sep="\t", row.names = F, col.names = T, quote = F)

### all samples to show J12 outlier

vsd = varianceStabilizingTransformation(dds, blind=FALSE)
sampleDists <- dist(t(assay(vsd)))

sampleDistMatrix <- as.matrix(sampleDists)
rownames(sampleDistMatrix) <- sample.df$name
colnames(sampleDistMatrix) <- NULL
colors <- colorRampPalette( rev(brewer.pal(9, "Blues")) )(255)

tiff(paste(path.fig, "/TDP43KD-mirna-cluster.tiff", sep=""), res=300, compression = "jpeg", width = 9, height = 9, unit = "in")
pheatmap(sampleDistMatrix,
         clustering_distance_rows=sampleDists,
         clustering_distance_cols=sampleDists,
         col=colors)

dev.off()

pdf(paste(path.fig, "/TDP43KD-mirna-cluster.pdf", sep=""))
pheatmap(sampleDistMatrix,
         clustering_distance_rows=sampleDists,
         clustering_distance_cols=sampleDists,
         col=colors)

dev.off()





### PCA


vsd = varianceStabilizingTransformation(dds, blind=FALSE)

plotPCA(vsd, intgroup=c("treatment", "replicate"), returnData = F)

pca.df = plotPCA(vsd, intgroup=c("treatment", "replicate"), returnData = TRUE)

pca.df$name = "Control KD"
pca.df$name[which(pca.df$treatment == "cds41")] = "TDP43 KD"

tiff(paste(path.fig, "/TDP43KD-mirna-PCA.tiff", sep=""), res=300, compression = "jpeg", width = 6, height = 6, unit = "in")

ggplot(pca.df) +
  aes(PC1, PC2, color = name) + 
  geom_point(size = 3) +   
  coord_fixed() +
  xlab("PC1: 42%")+ 
  ylab("PC2: 25%") +
  theme_bw() +
  theme(axis.text=element_text(size=16),
        axis.title = element_text(size=16),
        legend.position = "bottom",
        legend.title = element_blank(),
        legend.text = element_text(size=16),
        panel.grid.minor = element_blank()) +
  theme(aspect.ratio = 1)

dev.off()

pdf(paste(path.fig, "/TDP43KD-mirna-PCA.pdf", sep=""))

ggplot(pca.df) +
  aes(PC1, PC2, color = name) + 
  geom_point(size = 3) + 
  coord_fixed() +
  xlab("PC1: 37%")+ 
  ylab("PC2: 25%") +
  theme_bw() +
  theme(axis.text=element_text(size=16),
        axis.title = element_text(size=16),
        legend.text = element_text(size=16),
        legend.position = "bottom",
        legend.title = element_blank(),
        panel.grid.minor = element_blank()) +
  theme(aspect.ratio = 1)

dev.off()

#### volcano plot

df = read.table(paste(path.dir, "mirna-TDP43-KD-DESeq2.txt", sep="/"), sep="\t", header=T)

p.thresh = 0.01
fc.thresh = 0.58

up <-  df$GeneID[which(df$padj < p.thresh & df$log2FoldChange > fc.thresh)] 
down <-  df$GeneID[which(df$padj < p.thresh & df$log2FoldChange < -1*fc.thresh)] 

mycolors <- c("blue", "red", "gray")
names(mycolors) <- c("Down", "Up", "NC")

df$de <-  "NC"
df$de[match(up, df$GeneID)] <-  "Up"
df$de[match(down, df$GeneID)] <-  "Down"

df <- na.omit(df)

goi = c("hsa-miR-9-5p", "hsa-miR-9-3p", "hsa-miR-218-5p", "hsa-miR-124-5p", "hsa-miR-124-3p")
df.subset = df[match(goi, df$GeneID),]

p = ggplot(data=df) +
  geom_point(aes(x=log2FoldChange, y=-1*log10(padj), color=de, size=de)) + 
  labs(col=NULL) +
  geom_hline(yintercept= -log10(p.thresh), col="magenta") +
  scale_colour_manual(values = mycolors) +
  scale_size_manual(values = c(2, 1, 2)) +
  scale_x_continuous(limits = c(-2, 2), expand = c(0, 0))



p = p +  theme_bw() + 
  theme(legend.title=element_blank(), legend.text = element_text(colour="black", size = 20), legend.position = "bottom") +
  guides(colour = guide_legend(override.aes = list(size=4))) + 
  theme(axis.text.x = element_text(face="bold", color="black", size=16, angle=0)) +
  theme(axis.text.y = element_text(face="bold", color="black", size=16, angle=0)) +
  labs(x="log2(Fold change)", y="-1*log10(p-value)") +
  theme(axis.title = element_text(size = 20)) +
  theme(aspect.ratio = 1)


q = p +  geom_text_repel(data = df.subset, 
                         aes(x = log2FoldChange, y =  -1*log10(padj), label=GeneID), size = 6,
                         nudge_y=0.5, hjust=1.5, direction="x",
                         segment.color="green", segment.size=1.0,
                         min.segment.length = 0.2)

q

tiff(paste(path.fig, "/TDP43KD-mirna-volcano.tiff", sep=""), res=300, compression = "jpeg", width = 8, height = 8, unit = "in")
q
dev.off()

#########################################################################################################
### comparing aio with kd
library(fgsea)

aio.df = read.table( file = paste(path.dir, "mirna-aio-d40-DESeq2.txt", sep="/"), sep="\t", header = T)
kd.df = read.table( file = paste(path.dir, "mirna-TDP43-KD-DESeq2.txt", sep="/"), sep="\t", header = T)

p.thresh = 0.01
fc.thresh = 0.58

mir.up = aio.df$GeneID[which(aio.df$log2FoldChange > fc.thresh & aio.df$padj < p.thresh)]
mir.dw = aio.df$GeneID[which(aio.df$log2FoldChange < -1*fc.thresh & aio.df$padj < p.thresh)]

mir.ls = list()
mir.ls[["up"]] = mir.up
mir.ls[["dw"]] = mir.dw

rankvec = kd.df$stat
names(rankvec) = kd.df$GeneID
rankvec = sort(rankvec)

set.seed(12345)
res.fgsea = fgsea(mir.ls, rankvec, minSize = 15, maxSize = 1500, eps = 0, scoreType = "std")
(res.fgsea.df = data.frame(res.fgsea))

p.dw = plotEnrichment(mir.ls[["dw"]], rankvec)

p.dw = p.dw +
       geom_line(size = 2, col = "green") +
       theme_classic() +
       theme(panel.grid.minor = element_blank()) +
       theme(axis.text.x = element_text(face="bold", color="black", size=16, angle=0)) +
       theme(axis.text.y = element_text(face="bold", color="black", size=16, angle=0)) +
       theme(axis.title = element_blank()) +
       theme(axis.line = element_line(color = 'black', size = 1))
              
    
p.dw

tiff(paste(path.fig, "/gsea-aio-kd-dw.tiff", sep=""), res=300, compression = "jpeg", width = 8, height = 8, unit = "in")
p.dw
dev.off()


p.up = plotEnrichment(mir.ls[["up"]], rankvec)
p.up = p.up +
  geom_line(size = 2, col = "green") +
  theme_classic() +
  theme(panel.grid.minor = element_blank()) +
  theme(axis.text.x = element_text(face="bold", color="black", size=16, angle=0)) +
  theme(axis.text.y = element_text(face="bold", color="black", size=16, angle=0)) +
  theme(axis.title = element_blank()) +
  theme(axis.line = element_line(color = 'black', size = 1))

tiff(paste(path.fig, "/gsea-aio-kd-up.tiff", sep=""), res=300, compression = "jpeg", width = 8, height = 8, unit = "in")
p.up
dev.off()

write.table(res.fgsea.df[, 1:7], file = paste(path.fig, "/gsea-aio-kd.txt", sep=""), sep="\t", row.names = F, col.names = T, quote = F)
saveRDS(res.fgsea, paste(path.fig, "/gsea-aio-kd.rds", sep=""))

#####################################################################################################
#